<?php

class Database {
    private $pdo;

    public function connect() {
        if ($this->pdo == null) {
            try {
                $host     = $_ENV['DB_HOST'] ?? getenv('DB_HOST') ?: '127.0.0.1';
                $db_name  = $_ENV['DB_NAME'] ?? getenv('DB_NAME') ?: 'cargolist';
                $username = $_ENV['DB_USER'] ?? getenv('DB_USER') ?: 'cargolist_user';
                $password = $_ENV['DB_PASS'] ?? getenv('DB_PASS') ?: '';

                $dsn = "mysql:host=$host;dbname=$db_name;charset=utf8mb4";

                $this->pdo = new PDO($dsn, $username, $password);
                $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                $this->pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);

                $this->initTables();
            } catch (PDOException $e) {
                http_response_code(500);
                echo json_encode(['msg' => 'Database connection error: ' . $e->getMessage()]);
                exit;
            }
        }
        return $this->pdo;
    }

    private function initTables() {
        $queries = [
            "CREATE TABLE IF NOT EXISTS users (
                user_id       INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                email         VARCHAR(255) NOT NULL UNIQUE,
                password_hash VARCHAR(255) NOT NULL,
                first_name    VARCHAR(100) NOT NULL,
                last_name     VARCHAR(100) NOT NULL,
                phone_number  VARCHAR(30)  DEFAULT NULL,
                address       TEXT         DEFAULT NULL,
                email_verified TINYINT(1)  NOT NULL DEFAULT 0,
                creation_date DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB",

            "CREATE TABLE IF NOT EXISTS otps (
                otp_id        INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                user_id       INT UNSIGNED NOT NULL,
                email         VARCHAR(255) NOT NULL,
                code          VARCHAR(10)  NOT NULL,
                purpose       ENUM('register','reset_password') NOT NULL DEFAULT 'register',
                expire_date   DATETIME     NOT NULL,
                is_used       TINYINT(1)   NOT NULL DEFAULT 0,
                creation_date DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE
            ) ENGINE=InnoDB",

            "CREATE TABLE IF NOT EXISTS boats (
                boat_id             INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                owner_id            INT UNSIGNED NOT NULL,
                boat_name           VARCHAR(200) NOT NULL,
                registration_number VARCHAR(100) DEFAULT NULL,
                capacity            DECIMAL(10,2) DEFAULT 0,
                boat_type           VARCHAR(100) NOT NULL,
                daily_rate          DECIMAL(12,2) NOT NULL DEFAULT 0,
                image_url           TEXT         DEFAULT NULL,
                description         TEXT         DEFAULT NULL,
                length_m            DECIMAL(8,2) DEFAULT NULL,
                status              ENUM('available','unavailable','rented') NOT NULL DEFAULT 'available',
                creation_date       DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (owner_id) REFERENCES users(user_id) ON DELETE CASCADE
            ) ENGINE=InnoDB",

            "CREATE TABLE IF NOT EXISTS rentals (
                rental_id      INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
                renter_id      INT UNSIGNED NOT NULL,
                boat_id        INT UNSIGNED NOT NULL,
                start_date     DATE         NOT NULL,
                end_date       DATE         NOT NULL,
                total_cost     DECIMAL(14,2) NOT NULL DEFAULT 0,
                customer_name  VARCHAR(200) DEFAULT NULL,
                customer_email VARCHAR(255) DEFAULT NULL,
                customer_phone VARCHAR(30)  DEFAULT NULL,
                message        TEXT         DEFAULT NULL,
                status         ENUM('pending','confirmed','cancelled','completed') NOT NULL DEFAULT 'pending',
                creation_date  DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
                cargo_capacity DECIMAL(10,2) DEFAULT 0,
                FOREIGN KEY (renter_id) REFERENCES users(user_id) ON DELETE CASCADE,
                FOREIGN KEY (boat_id)   REFERENCES boats(boat_id)  ON DELETE CASCADE
            ) ENGINE=InnoDB",
        ];

        foreach ($queries as $query) {
            $this->pdo->exec($query);
        }

        // Idempotent column migrations
        $this->pdo->exec(
            "ALTER TABLE rentals ADD COLUMN IF NOT EXISTS
             cargo_capacity DECIMAL(10,2) NOT NULL DEFAULT 0"
        );
    }
}
