<?php
// When used as PHP built-in server router, serve static files (uploads, etc.) natively
if (php_sapi_name() === 'cli-server') {
    $file = __DIR__ . parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    if (is_file($file)) {
        return false; // let PHP serve the file directly
    }
}

require_once __DIR__ . '/vendor/autoload.php';
require_once __DIR__ . '/config/database.php';

// Load .env
$dotenv = Dotenv\Dotenv::createImmutable(__DIR__);
$dotenv->load();

// CORS
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// DB
$database = new Database();
$pdo      = $database->connect();

// Classes
require_once __DIR__ . '/src/Auth.php';
require_once __DIR__ . '/src/Boat.php';
require_once __DIR__ . '/src/Rental.php';

$auth   = new Auth($pdo);
$boat   = new Boat($pdo);
$rental = new Rental($pdo);

function getJsonInput() {
    return json_decode(file_get_contents("php://input"), true) ?? [];
}

// Router
$uri    = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$method = $_SERVER['REQUEST_METHOD'];

$basePath = dirname($_SERVER['SCRIPT_NAME']);
$path = str_replace($basePath, '', $uri);
// Strip /api prefix if present (e.g. dev proxy or nginx sub-path)
# $path = preg_replace('#^/api#', '', $uri);
// Also strip trailing slash
# $path = rtrim($path, '/') ?: '/';

// Extract dynamic segments: /boats/42  ->  ['boats', '42']
$segments = explode('/', trim($path, '/'));

try {
    // ── AUTH ──────────────────────────────────────────────────────────────────
    if ($path === '/auth/register' && $method === 'POST') {
        echo json_encode($auth->register(getJsonInput()));

    } elseif ($path === '/auth/login' && $method === 'POST') {
        echo json_encode($auth->login(getJsonInput()));

    } elseif ($path === '/auth/verify-otp' && $method === 'POST') {
        echo json_encode($auth->verifyOTP(getJsonInput()));

    } elseif ($path === '/auth/resend-otp' && $method === 'POST') {
        echo json_encode($auth->resendOTP(getJsonInput()));

    } elseif ($path === '/auth/forgot-password' && $method === 'POST') {
        echo json_encode($auth->requestPasswordReset(getJsonInput()));

    } elseif ($path === '/auth/reset-password' && $method === 'POST') {
        echo json_encode($auth->resetPassword(getJsonInput()));

    } elseif ($path === '/auth/profile' && $method === 'GET') {
        $userId = $auth->authenticate();
        echo json_encode($auth->getUser($userId));

    } elseif ($path === '/auth/profile' && $method === 'PUT') {
        $userId = $auth->authenticate();
        echo json_encode($auth->updateProfile($userId, getJsonInput()));

    // ── BOATS ─────────────────────────────────────────────────────────────────
    } elseif ($path === '/boats/available' && $method === 'GET') {
        // Public — available boats for renters (no auth needed)
        echo json_encode($boat->listAvailable());

    } elseif ($path === '/boats/upload-image' && $method === 'POST') {
        // Upload a boat photo — returns { url: "..." }
        $userId = $auth->authenticate();
        if (empty($_FILES['image'])) {
            throw new Exception("No image file provided", 400);
        }
        $file     = $_FILES['image'];
        $allowed  = ['image/jpeg', 'image/png', 'image/webp', 'image/gif'];
        $mime     = mime_content_type($file['tmp_name']);
        if (!in_array($mime, $allowed)) {
            throw new Exception("Invalid file type. Only JPG, PNG, WEBP, GIF are allowed.", 400);
        }
        if ($file['size'] > 5 * 1024 * 1024) {
            throw new Exception("File too large. Max size is 5 MB.", 400);
        }
        $ext      = pathinfo($file['name'], PATHINFO_EXTENSION) ?: explode('/', $mime)[1];
        $filename = $userId . '_' . uniqid() . '.' . strtolower($ext);
        $destDir  = __DIR__ . '/uploads/';
        $dest     = $destDir . $filename;
		
		if (!move_uploaded_file($file['tmp_name'], $dest)) {
			throw new Exception("Failed to save image", 500);
		}

		// 1. Get Protocol (http/https)
		$protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';

		// 2. Get Host (eliazar.heliohost.us)
		$host = $_SERVER['HTTP_HOST'];

		// 3. Get the web folder path (/cargolist/backend)
		// dirname($_SERVER['SCRIPT_NAME']) returns the path to the folder containing this index.php
		$path = dirname($_SERVER['SCRIPT_NAME']); 

		// 4. Construct the correct URL
		// Result: http://eliazar.heliohost.us/cargolist/backend/uploads/filename.jpg
		echo json_encode(['url' => "$protocol://$host$path/uploads/$filename"]);

    } elseif ($path === '/boats' && $method === 'GET') {
        // Owner's fleet
        $userId = $auth->authenticate();
        echo json_encode($boat->listByOwner($userId));

    } elseif ($path === '/boats' && $method === 'POST') {
        $userId = $auth->authenticate();
        echo json_encode($boat->create($userId, getJsonInput()));

    } elseif (isset($segments[0], $segments[1], $segments[2])
        && $segments[0] === 'boats' && is_numeric($segments[1])
        && $segments[2] === 'available-capacity' && $method === 'GET') {
        // Public — live capacity check for a boat + date range
        $boatId    = (int)$segments[1];
        $startDate = trim($_GET['start_date'] ?? '');
        $endDate   = trim($_GET['end_date']   ?? '');
        if (!$startDate || !$endDate) {
            throw new Exception("start_date and end_date query params are required", 400);
        }
        echo json_encode($rental->getAvailableCapacity($boatId, $startDate, $endDate));

    } elseif (isset($segments[0], $segments[1]) && $segments[0] === 'boats' && is_numeric($segments[1])) {
        $boatId = (int)$segments[1];

        if ($method === 'GET') {
            echo json_encode($boat->getOne($boatId));

        } elseif ($method === 'PUT') {
            $userId = $auth->authenticate();
            echo json_encode($boat->update($boatId, $userId, getJsonInput()));

        } elseif ($method === 'DELETE') {
            $userId = $auth->authenticate();
            echo json_encode($boat->delete($boatId, $userId));

        } else {
            http_response_code(405);
            echo json_encode(['msg' => 'Method not allowed']);
        }

    // ── RENTALS ───────────────────────────────────────────────────────────────
    } elseif ($path === '/rentals' && $method === 'POST') {
        $userId = $auth->authenticate();
        echo json_encode($rental->create($userId, getJsonInput()));

    } elseif ($path === '/rentals' && $method === 'GET') {
        // Owner sees bookings on their boats
        $userId = $auth->authenticate();
        echo json_encode($rental->listByOwner($userId));

    } elseif ($path === '/rentals/my' && $method === 'GET') {
        // Renter sees their own bookings
        $userId = $auth->authenticate();
        echo json_encode($rental->listByRenter($userId));

    } elseif (isset($segments[0], $segments[1], $segments[2])
        && $segments[0] === 'rentals' && is_numeric($segments[1]) && $segments[2] === 'status'
        && $method === 'PUT') {
        $userId   = $auth->authenticate();
        $rentalId = (int)$segments[1];
        echo json_encode($rental->updateStatus($rentalId, $userId, getJsonInput()));

    // ── FALLBACK ──────────────────────────────────────────────────────────────
    } else {
        http_response_code(404);
        echo json_encode(['msg' => "Route not SDF found: $method $path"]);
    }

} catch (Exception $e) {
    $code = (int)$e->getCode();
    http_response_code($code >= 100 && $code < 600 ? $code : 500);
    echo json_encode(['msg' => $e->getMessage()]);
}
