<?php

class Boat {
    private $pdo;

    public function __construct($pdo) {
        $this->pdo = $pdo;
    }

    // POST /boats — owner creates a boat
    public function create($ownerId, $data) {
        $boat_name  = trim($data['boat_name']           ?? '');
        $boat_type  = trim($data['boat_type']           ?? '');
        $daily_rate = $data['daily_rate']               ?? 0;
        $reg_no     = trim($data['registration_number'] ?? '');
        $capacity   = $data['capacity']                 ?? 0;
        $image_url  = trim($data['image_url']           ?? '');
        $description = trim($data['description']        ?? '');
        $length_m   = $data['length_m']                 ?? null;

        if (!$boat_name || !$boat_type || !$daily_rate) {
            throw new Exception("Boat name, type, and daily rate are required", 400);
        }

        $stmt = $this->pdo->prepare(
            "INSERT INTO boats (owner_id, boat_name, registration_number, capacity, boat_type,
                                daily_rate, image_url, description, length_m)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)"
        );
        $stmt->execute([
            $ownerId, $boat_name, $reg_no ?: null, $capacity, $boat_type,
            $daily_rate, $image_url ?: null, $description ?: null, $length_m,
        ]);

        return $this->getOne((int)$this->pdo->lastInsertId(), $ownerId);
    }

    // GET /boats — owner's fleet
    public function listByOwner($ownerId) {
        $stmt = $this->pdo->prepare(
            "SELECT b.*, u.first_name, u.last_name
             FROM boats b JOIN users u ON u.user_id = b.owner_id
             WHERE b.owner_id = ?
             ORDER BY b.creation_date DESC"
        );
        $stmt->execute([$ownerId]);
        return $stmt->fetchAll();
    }

    // GET /boats/available — all available boats for renters
    public function listAvailable() {
        $stmt = $this->pdo->query(
            "SELECT b.*, u.first_name, u.last_name
             FROM boats b JOIN users u ON u.user_id = b.owner_id
             WHERE b.status = 'available'
             ORDER BY b.creation_date DESC"
        );
        return $stmt->fetchAll();
    }

    // GET /boats/:id
    public function getOne($boatId, $ownerId = null) {
        $sql  = "SELECT b.*, u.first_name, u.last_name
                 FROM boats b JOIN users u ON u.user_id = b.owner_id
                 WHERE b.boat_id = ?";
        $params = [$boatId];

        if ($ownerId !== null) {
            $sql   .= " AND b.owner_id = ?";
            $params[] = $ownerId;
        }

        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        $boat = $stmt->fetch();
        if (!$boat) throw new Exception("Boat not found", 404);
        return $boat;
    }

    // PUT /boats/:id — owner updates their boat
    public function update($boatId, $ownerId, $data) {
        $this->getOne($boatId, $ownerId); // ensures ownership

        $boat_name   = trim($data['boat_name']           ?? '');
        $boat_type   = trim($data['boat_type']           ?? '');
        $daily_rate  = $data['daily_rate']               ?? 0;
        $reg_no      = trim($data['registration_number'] ?? '');
        $capacity    = $data['capacity']                 ?? 0;
        $image_url   = trim($data['image_url']           ?? '');
        $description = trim($data['description']         ?? '');
        $length_m    = $data['length_m']                 ?? null;
        $status      = $data['status']                   ?? 'available';

        if (!in_array($status, ['available', 'unavailable', 'rented'])) {
            throw new Exception("Invalid status", 400);
        }

        $stmt = $this->pdo->prepare(
            "UPDATE boats SET boat_name=?, boat_type=?, daily_rate=?, registration_number=?,
                              capacity=?, image_url=?, description=?, length_m=?, status=?
             WHERE boat_id=? AND owner_id=?"
        );
        $stmt->execute([
            $boat_name, $boat_type, $daily_rate, $reg_no ?: null,
            $capacity, $image_url ?: null, $description ?: null, $length_m, $status,
            $boatId, $ownerId,
        ]);

        return $this->getOne($boatId, $ownerId);
    }

    // DELETE /boats/:id
    public function delete($boatId, $ownerId) {
        $this->getOne($boatId, $ownerId); // ensures ownership
        $stmt = $this->pdo->prepare("DELETE FROM boats WHERE boat_id = ? AND owner_id = ?");
        $stmt->execute([$boatId, $ownerId]);
        return ['msg' => 'Boat removed'];
    }
}
