<?php

class Rental {
    private $pdo;

    public function __construct($pdo) {
        $this->pdo = $pdo;
    }

    // POST /rentals — renter submits a rental request
    public function create($renterId, $data) {
        $boat_id        = (int)($data['boat_id']        ?? 0);
        $start_date     = trim($data['start_date']      ?? '');
        $end_date       = trim($data['end_date']        ?? '');
        $customer_name  = trim($data['customer_name']   ?? '');
        $customer_email = trim($data['customer_email']  ?? '');
        $customer_phone = trim($data['customer_phone']  ?? '');
        $message        = trim($data['message']         ?? '');
        $cargo_capacity = (float)($data['cargo_capacity'] ?? 0);

        if (!$boat_id || !$start_date || !$end_date) {
            throw new Exception("Boat, start date, and end date are required", 400);
        }
        if ($cargo_capacity <= 0) {
            throw new Exception("Cargo capacity must be greater than 0", 400);
        }

        // Fetch boat details
        $stmt = $this->pdo->prepare("SELECT boat_id, daily_rate, capacity, status FROM boats WHERE boat_id = ?");
        $stmt->execute([$boat_id]);
        $boat = $stmt->fetch();
        if (!$boat) throw new Exception("Boat not found", 404);
        if ($boat['status'] === 'unavailable') throw new Exception("Boat is not available", 409);

        // Check capacity: sum of cargo_capacity for non-cancelled, overlapping rentals
        $stmt = $this->pdo->prepare(
            "SELECT COALESCE(SUM(cargo_capacity), 0) AS used_capacity
             FROM rentals
             WHERE boat_id = ?
               AND status NOT IN ('cancelled', 'completed')
               AND start_date <= ?
               AND end_date   >= ?"
        );
        $stmt->execute([$boat_id, $end_date, $start_date]);
        $used = (float)$stmt->fetchColumn();

        if ($used + $cargo_capacity > (float)$boat['capacity']) {
            $remaining = max(0, (float)$boat['capacity'] - $used);
            throw new Exception(
                "Not enough cargo capacity available. Only " .
                number_format($remaining, 2) . " tons remaining for those dates.",
                409
            );
        }

        $days = max(1, (int)ceil(
            (strtotime($end_date) - strtotime($start_date)) / 86400
        ));
        $total_cost = round($boat['daily_rate'] * $days * $cargo_capacity, 2);

        $stmt = $this->pdo->prepare(
            "INSERT INTO rentals
                (renter_id, boat_id, start_date, end_date, total_cost,
                 customer_name, customer_email, customer_phone, message, cargo_capacity)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)"
        );
        $stmt->execute([
            $renterId, $boat_id, $start_date, $end_date, $total_cost,
            $customer_name ?: null, $customer_email ?: null,
            $customer_phone ?: null, $message ?: null,
            $cargo_capacity,
        ]);

        return $this->getOne((int)$this->pdo->lastInsertId());
    }

    // GET /rentals — bookings ON the owner's boats
    public function listByOwner($ownerId) {
        $stmt = $this->pdo->prepare(
            "SELECT r.*, b.boat_name, b.boat_type, b.daily_rate, b.capacity AS boat_capacity, b.image_url,
                    u.first_name AS renter_first, u.last_name AS renter_last, u.email AS renter_email
             FROM rentals r
             JOIN boats b ON b.boat_id = r.boat_id
             JOIN users u ON u.user_id = r.renter_id
             WHERE b.owner_id = ?
             ORDER BY r.creation_date DESC"
        );
        $stmt->execute([$ownerId]);
        return $stmt->fetchAll();
    }

    // GET /rentals/my — rentals made BY the renter
    public function listByRenter($renterId) {
        $stmt = $this->pdo->prepare(
            "SELECT r.*, b.boat_name, b.boat_type, b.daily_rate, b.capacity AS boat_capacity, b.image_url,
                    u.first_name AS owner_first, u.last_name AS owner_last
             FROM rentals r
             JOIN boats b  ON b.boat_id  = r.boat_id
             JOIN users u  ON u.user_id  = b.owner_id
             WHERE r.renter_id = ?
             ORDER BY r.creation_date DESC"
        );
        $stmt->execute([$renterId]);
        return $stmt->fetchAll();
    }

    // GET /rentals/:id
    public function getOne($rentalId) {
        $stmt = $this->pdo->prepare(
            "SELECT r.*, b.boat_name, b.boat_type, b.capacity AS boat_capacity, b.image_url
             FROM rentals r JOIN boats b ON b.boat_id = r.boat_id
             WHERE r.rental_id = ?"
        );
        $stmt->execute([$rentalId]);
        $rental = $stmt->fetch();
        if (!$rental) throw new Exception("Rental not found", 404);
        return $rental;
    }

    // GET /boats/:id/available-capacity — public endpoint for live capacity check
    public function getAvailableCapacity($boatId, $startDate, $endDate) {
        $stmt = $this->pdo->prepare(
            "SELECT capacity FROM boats WHERE boat_id = ?"
        );
        $stmt->execute([(int)$boatId]);
        $boat = $stmt->fetch();
        if (!$boat) throw new Exception("Boat not found", 404);

        $total = (float)$boat['capacity'];

        $stmt = $this->pdo->prepare(
            "SELECT COALESCE(SUM(cargo_capacity), 0) AS used_capacity
             FROM rentals
             WHERE boat_id = ?
               AND status NOT IN ('cancelled', 'completed')
               AND start_date <= ?
               AND end_date   >= ?"
        );
        $stmt->execute([(int)$boatId, $endDate, $startDate]);
        $used = (float)$stmt->fetchColumn();

        return [
            'boat_id'   => (int)$boatId,
            'total'     => $total,
            'used'      => round($used, 2),
            'remaining' => round(max(0, $total - $used), 2),
        ];
    }

    // PUT /rentals/:id/status — owner updates status (confirm/cancel/complete)
    public function updateStatus($rentalId, $ownerId, $data) {
        $status = $data['status'] ?? '';
        if (!in_array($status, ['confirmed', 'cancelled', 'completed'])) {
            throw new Exception("Invalid status. Use: confirmed, cancelled, completed", 400);
        }

        // Ensure the rental is on one of the owner's boats
        $stmt = $this->pdo->prepare(
            "SELECT r.rental_id FROM rentals r
             JOIN boats b ON b.boat_id = r.boat_id
             WHERE r.rental_id = ? AND b.owner_id = ?"
        );
        $stmt->execute([$rentalId, $ownerId]);
        if (!$stmt->fetch()) throw new Exception("Rental not found or access denied", 404);

        $stmt = $this->pdo->prepare("UPDATE rentals SET status = ? WHERE rental_id = ?");
        $stmt->execute([$status, $rentalId]);

        return $this->getOne($rentalId);
    }
}
